<?php

namespace Hrm\Unit\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Hrm\Unit\Http\Requests\UnitRequest;
use Hrm\Unit\Interfaces\UnitRepositoryInterface;
use Hrm\Unit\Models\Unit;

/**
 * Resource controller class for unit.
 */
class UnitResourceController extends BaseController
{

    /**
     * Initialize unit resource controller.
     *
     * @param type UnitRepositoryInterface $unit
     *
     * @return null
     */
    public function __construct(UnitRepositoryInterface $unit)
    {
        parent::__construct();
        $this->repository = $unit;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Hrm\Unit\Repositories\Criteria\UnitResourceCriteria::class);
    }

    /**
     * Display a list of unit.
     *
     * @return Response
     */
    public function index(UnitRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Hrm\Unit\Repositories\Presenter\UnitPresenter::class)
                ->$function();
        }

        $units = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('unit::unit.names'))
            ->view('unit::unit.index', true)
            ->data(compact('units', 'view'))
            ->output();
    }

    /**
     * Display unit.
     *
     * @param Request $request
     * @param Model   $unit
     *
     * @return Response
     */
    public function show(UnitRequest $request, Unit $unit)
    {

        if ($unit->exists) {
            $view = 'unit::unit.show';
        } else {
            $view = 'unit::unit.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('unit::unit.name'))
            ->data(compact('unit'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new unit.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(UnitRequest $request)
    {

        $unit = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('unit::unit.name')) 
            ->view('unit::unit.create', true) 
            ->data(compact('unit'))
            ->output();
    }

    /**
     * Create new unit.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(UnitRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $unit                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('unit::unit.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('unit/unit/' . $unit->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/unit/unit'))
                ->redirect();
        }

    }

    /**
     * Show unit for editing.
     *
     * @param Request $request
     * @param Model   $unit
     *
     * @return Response
     */
    public function edit(UnitRequest $request, Unit $unit)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('unit::unit.name'))
            ->view('unit::unit.edit', true)
            ->data(compact('unit'))
            ->output();
    }

    /**
     * Update the unit.
     *
     * @param Request $request
     * @param Model   $unit
     *
     * @return Response
     */
    public function update(UnitRequest $request, Unit $unit)
    {
        try {
            $attributes = $request->all();

            $unit->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('unit::unit.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('unit/unit/' . $unit->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('unit/unit/' . $unit->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the unit.
     *
     * @param Model   $unit
     *
     * @return Response
     */
    public function destroy(UnitRequest $request, Unit $unit)
    {
        try {

            $unit->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('unit::unit.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('unit/unit/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('unit/unit/' . $unit->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple unit.
     *
     * @param Model   $unit
     *
     * @return Response
     */
    public function delete(UnitRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('unit::unit.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('unit/unit'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/unit/unit'))
                ->redirect();
        }

    }

    /**
     * Restore deleted units.
     *
     * @param Model   $unit
     *
     * @return Response
     */
    public function restore(UnitRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('unit::unit.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/unit/unit'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/unit/unit/'))
                ->redirect();
        }

    }

}
