<?php

namespace Ebuy\Question\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Repository\Filter\RequestFilter;
use Ebuy\Question\Forms\Question as QuestionForm;
use Ebuy\Question\Http\Requests\QuestionRequest;
use Ebuy\Question\Interfaces\QuestionRepositoryInterface;
use Ebuy\Question\Repositories\Eloquent\Filters\QuestionResourceFilter;
use Ebuy\Question\Repositories\Eloquent\Presenters\QuestionListPresenter;

/**
 * Resource controller class for question.
 */
class QuestionResourceController extends BaseController
{

    /**
     * Initialize question resource controller.
     *
     *
     * @return null
     */
    public function __construct(QuestionRepositoryInterface $question)
    {
        parent::__construct();
        $this->form = QuestionForm::setAttributes()->toArray();
        $this->modules = $this->modules(config('ebuy.question.modules'), 'question', guard_url('question'));
        $this->repository = $question;
    }

    /**
     * Display a list of question.
     *
     * @return Response
     */
    public function index(QuestionRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $data = $this->repository
            ->pushFilter(RequestFilter::class)
            ->pushFilter(QuestionResourceFilter::class)
            ->setPresenter(QuestionListPresenter::class)
            ->simplePaginate($pageLimit)
            // ->withQueryString()
            ->toArray();

        extract($data);
        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('question::question.names'))
            ->view('question::question.index')
            ->data(compact('data', 'meta', 'links', 'modules', 'form'))
            ->output();
    }

    /**
     * Display question.
     *
     * @param Request $request
     * @param Model   $question
     *
     * @return Response
     */
    public function show(QuestionRequest $request, QuestionRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('question::question.name'))
            ->data(compact('data', 'form', 'modules', 'form'))
            ->view('question::question.show')
            ->output();
    }

    /**
     * Show the form for creating a new question.
     *
     * @param Request $request
     *x
     * @return Response
     */
    public function create(QuestionRequest $request, QuestionRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('question::question.name'))
            ->view('question::question.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Create new question.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(QuestionRequest $request, QuestionRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $repository->create($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.created', ['Module' => trans('question::question.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('question/question/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/question/question'))
                ->redirect();
        }

    }

    /**
     * Show question for editing.
     *
     * @param Request $request
     * @param Model   $question
     *
     * @return Response
     */
    public function edit(QuestionRequest $request, QuestionRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('question::question.name'))
            ->view('question::question.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Update the question.
     *
     * @param Request $request
     * @param Model   $question
     *
     * @return Response
     */
    public function update(QuestionRequest $request, QuestionRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $repository->update($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('question::question.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('question/question/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('question/question/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the question.
     *
     * @param Model   $question
     *
     * @return Response
     */
    public function destroy(QuestionRequest $request, QuestionRepositoryInterface $repository)
    {
        try {
            $repository->delete();
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('question::question.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('question/question/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('question/question/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }
}
