<?php

namespace Ebuy\Question\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Repository\Filter\RequestFilter;
use Ebuy\Question\Forms\Faq as FaqForm;
use Ebuy\Question\Http\Requests\FaqRequest;
use Ebuy\Question\Interfaces\FaqRepositoryInterface;
use Ebuy\Question\Repositories\Eloquent\Filters\FaqResourceFilter;
use Ebuy\Question\Repositories\Eloquent\Presenters\FaqListPresenter;

/**
 * Resource controller class for faq.
 */
class FaqResourceController extends BaseController
{

    /**
     * Initialize faq resource controller.
     *
     *
     * @return null
     */
    public function __construct(FaqRepositoryInterface $faq)
    {
        parent::__construct();
        $this->form = FaqForm::setAttributes()->toArray();
        $this->modules = $this->modules(config('ebuy.question.modules'), 'question', guard_url('question'));
        $this->repository = $faq;
    }

    /**
     * Display a list of faq.
     *
     * @return Response
     */
    public function index(FaqRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $data = $this->repository
            ->pushFilter(RequestFilter::class)
            ->pushFilter(FaqResourceFilter::class)
            ->setPresenter(FaqListPresenter::class)
            ->simplePaginate($pageLimit)
            // ->withQueryString()
            ->toArray();

        extract($data);
        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('question::faq.names'))
            ->view('question::faq.index')
            ->data(compact('data', 'meta', 'links', 'modules', 'form'))
            ->output();
    }

    /**
     * Display faq.
     *
     * @param Request $request
     * @param Model   $faq
     *
     * @return Response
     */
    public function show(FaqRequest $request, FaqRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('question::faq.name'))
            ->data(compact('data', 'form', 'modules', 'form'))
            ->view('question::faq.show')
            ->output();
    }

    /**
     * Show the form for creating a new faq.
     *
     * @param Request $request
     *x
     * @return Response
     */
    public function create(FaqRequest $request, FaqRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('question::faq.name'))
            ->view('question::faq.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Create new faq.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(FaqRequest $request, FaqRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $repository->create($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.created', ['Module' => trans('question::faq.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('question/faq/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/question/faq'))
                ->redirect();
        }

    }

    /**
     * Show faq for editing.
     *
     * @param Request $request
     * @param Model   $faq
     *
     * @return Response
     */
    public function edit(FaqRequest $request, FaqRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('question::faq.name'))
            ->view('question::faq.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Update the faq.
     *
     * @param Request $request
     * @param Model   $faq
     *
     * @return Response
     */
    public function update(FaqRequest $request, FaqRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $repository->update($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('question::faq.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('question/faq/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('question/faq/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the faq.
     *
     * @param Model   $faq
     *
     * @return Response
     */
    public function destroy(FaqRequest $request, FaqRepositoryInterface $repository)
    {
        try {
            $repository->delete();
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('question::faq.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('question/faq/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('question/faq/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }
}
