<?php

namespace Order\Eee\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Database\RequestScope;
use Order\Eee\Forms\Subscription as SubscriptionForm;
use Order\Eee\Http\Requests\SubscriptionRequest;
use Order\Eee\Http\Resources\SubscriptionResource;
use Order\Eee\Http\Resources\SubscriptionsCollection;
use Order\Eee\Models\Subscription;
use Order\Eee\Scopes\SubscriptionResourceScope;


/**
 * Resource controller class for subscription.
 */
class SubscriptionResourceController extends BaseController
{

    /**
     * Initialize subscription resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->form = SubscriptionForm::grouped(false)
                        ->setAttributes()
                        ->toArray();
        $this->modules = $this->modules(config('order.eee.modules'), 'eee', guard_url('eee'));
    }

    /**
     * Display a list of subscription.
     *
     * @return Response
     */
    public function index(SubscriptionRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $page = Subscription::pushScope(new RequestScope())
            ->pushScope(new SubscriptionResourceScope())
            ->simplePaginate($pageLimit);

        $data = new SubscriptionsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('eee::subscription.names'))
            ->view('eee::subscription.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display subscription.
     *
     * @param Request $request
     * @param Model   $subscription
     *
     * @return Response
     */
    public function show(SubscriptionRequest $request, Subscription $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SubscriptionResource($model);

        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('eee::subscription.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('eee::subscription.show')
            ->output();

    }

    /**
     * Show the form for creating a new subscription.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(SubscriptionRequest $request, Subscription $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SubscriptionResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('eee::subscription.name'))
            ->view('eee::subscription.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new subscription.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(SubscriptionRequest $request, Subscription $model)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $model = $model->create($attributes);
            $data = new SubscriptionResource($model);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('eee::subscription.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('eee/subscription/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/eee/subscription'))
                ->redirect();
        }

    }

    /**
     * Show subscription for editing.
     *
     * @param Request $request
     * @param Model   $subscription
     *
     * @return Response
     */
    public function edit(SubscriptionRequest $request, Subscription $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SubscriptionResource($model);

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('eee::subscription.name'))
            ->view('eee::subscription.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the subscription.
     *
     * @param Request $request
     * @param Model   $subscription
     *
     * @return Response
     */
    public function update(SubscriptionRequest $request, Subscription $model)
    {
        try {
            $attributes = $request->all();
            $model = $model->update($attributes);
            $data = new SubscriptionResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('eee::subscription.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('eee/subscription/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('eee/subscription/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the subscription.
     *
     * @param Model   $subscription
     *
     * @return Response
     */
    public function destroy(SubscriptionRequest $request, Subscription $model)
    {
        try {
            $model->delete();
            $data = new SubscriptionResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('eee::subscription.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('eee/subscription/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('eee/subscription/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
