<?php

namespace Litecms\Newsletter\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Litecms\Newsletter\Http\Requests\NewsletterRequest;
use Litecms\Newsletter\Interfaces\NewsletterRepositoryInterface;
use Litecms\Newsletter\Models\Newsletter;

/**
 * Resource controller class for newsletter.
 */
class NewsletterResourceController extends BaseController
{

    /**
     * Initialize newsletter resource controller.
     *
     * @param type NewsletterRepositoryInterface $newsletter
     *
     * @return null
     */
    public function __construct(NewsletterRepositoryInterface $newsletter)
    {
        parent::__construct();
        $this->repository = $newsletter;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Litecms\Newsletter\Repositories\Criteria\NewsletterResourceCriteria::class);
    }

    /**
     * Display a list of newsletter.
     *
     * @return Response
     */
    public function index(NewsletterRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Litecms\Newsletter\Repositories\Presenter\NewsletterPresenter::class)
                ->$function();
        }

        $newsletters = $this->repository->paginate();

        return $this->response->title(trans('newsletter::newsletter.names'))
            ->view('newsletter::newsletter.index', true)
            ->data(compact('newsletters'))
            ->output();
    }

    /**
     * Display newsletter.
     *
     * @param Request $request
     * @param Model   $newsletter
     *
     * @return Response
     */
    public function show(NewsletterRequest $request, Newsletter $newsletter)
    {

        if ($newsletter->exists) {
            $view = 'newsletter::newsletter.show';
        } else {
            $view = 'newsletter::newsletter.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('newsletter::newsletter.name'))
            ->data(compact('newsletter'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new newsletter.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(NewsletterRequest $request)
    {

        $newsletter = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('newsletter::newsletter.name')) 
            ->view('newsletter::newsletter.create', true) 
            ->data(compact('newsletter'))
            ->output();
    }

    /**
     * Create new newsletter.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(NewsletterRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $newsletter                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('newsletter::newsletter.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('newsletter/newsletter/' . $newsletter->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/newsletter/newsletter'))
                ->redirect();
        }

    }

    /**
     * Show newsletter for editing.
     *
     * @param Request $request
     * @param Model   $newsletter
     *
     * @return Response
     */
    public function edit(NewsletterRequest $request, Newsletter $newsletter)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('newsletter::newsletter.name'))
            ->view('newsletter::newsletter.edit', true)
            ->data(compact('newsletter'))
            ->output();
    }

    /**
     * Update the newsletter.
     *
     * @param Request $request
     * @param Model   $newsletter
     *
     * @return Response
     */
    public function update(NewsletterRequest $request, Newsletter $newsletter)
    {
        try {
            $attributes = $request->all();

            $newsletter->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('newsletter::newsletter.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('newsletter/newsletter/' . $newsletter->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('newsletter/newsletter/' . $newsletter->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the newsletter.
     *
     * @param Model   $newsletter
     *
     * @return Response
     */
    public function destroy(NewsletterRequest $request, Newsletter $newsletter)
    {
        try {

            $newsletter->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('newsletter::newsletter.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('newsletter/newsletter/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('newsletter/newsletter/' . $newsletter->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple newsletter.
     *
     * @param Model   $newsletter
     *
     * @return Response
     */
    public function delete(NewsletterRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('newsletter::newsletter.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('newsletter/newsletter'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/newsletter/newsletter'))
                ->redirect();
        }

    }

    /**
     * Restore deleted newsletters.
     *
     * @param Model   $newsletter
     *
     * @return Response
     */
    public function restore(NewsletterRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('newsletter::newsletter.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/newsletter/newsletter'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/newsletter/newsletter/'))
                ->redirect();
        }

    }

}
