<?php

namespace Litecms\Mcq\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Litecms\Mcq\Http\Requests\McqRequest;
use Litecms\Mcq\Interfaces\McqRepositoryInterface;
use Litecms\Mcq\Models\Mcq;

/**
 * Resource controller class for mcq.
 */
class McqResourceController extends BaseController
{

    /**
     * Initialize mcq resource controller.
     *
     * @param type McqRepositoryInterface $mcq
     *
     * @return null
     */
    public function __construct(McqRepositoryInterface $mcq)
    {
        parent::__construct();
        $this->repository = $mcq;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Litecms\Mcq\Repositories\Criteria\McqResourceCriteria::class);
    }

    /**
     * Display a list of mcq.
     *
     * @return Response
     */
    public function index(McqRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Litecms\Mcq\Repositories\Presenter\McqPresenter::class)
                ->$function();
        }

        $mcqs = $this->repository->paginate();

        return $this->response->title(trans('mcq::mcq.names'))
            ->view('mcq::mcq.index', true)
            ->data(compact('mcqs'))
            ->output();
    }

    /**
     * Display mcq.
     *
     * @param Request $request
     * @param Model   $mcq
     *
     * @return Response
     */
    public function show(McqRequest $request, Mcq $mcq)
    {

        if ($mcq->exists) {
            $view = 'mcq::mcq.show';
        } else {
            $view = 'mcq::mcq.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('mcq::mcq.name'))
            ->data(compact('mcq'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new mcq.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(McqRequest $request)
    {

        $mcq = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('mcq::mcq.name')) 
            ->view('mcq::mcq.create', true) 
            ->data(compact('mcq'))
            ->output();
    }

    /**
     * Create new mcq.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(McqRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $mcq                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('mcq::mcq.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('mcq/mcq/' . $mcq->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/mcq/mcq'))
                ->redirect();
        }

    }

    /**
     * Show mcq for editing.
     *
     * @param Request $request
     * @param Model   $mcq
     *
     * @return Response
     */
    public function edit(McqRequest $request, Mcq $mcq)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('mcq::mcq.name'))
            ->view('mcq::mcq.edit', true)
            ->data(compact('mcq'))
            ->output();
    }

    /**
     * Update the mcq.
     *
     * @param Request $request
     * @param Model   $mcq
     *
     * @return Response
     */
    public function update(McqRequest $request, Mcq $mcq)
    {
        try {
            $attributes = $request->all();

            $mcq->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('mcq::mcq.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('mcq/mcq/' . $mcq->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('mcq/mcq/' . $mcq->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the mcq.
     *
     * @param Model   $mcq
     *
     * @return Response
     */
    public function destroy(McqRequest $request, Mcq $mcq)
    {
        try {

            $mcq->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('mcq::mcq.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('mcq/mcq/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('mcq/mcq/' . $mcq->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple mcq.
     *
     * @param Model   $mcq
     *
     * @return Response
     */
    public function delete(McqRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('mcq::mcq.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('mcq/mcq'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/mcq/mcq'))
                ->redirect();
        }

    }

    /**
     * Restore deleted mcqs.
     *
     * @param Model   $mcq
     *
     * @return Response
     */
    public function restore(McqRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('mcq::mcq.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/mcq/mcq'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/mcq/mcq/'))
                ->redirect();
        }

    }

}
