<?php

namespace Survey\Language\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Survey\Language\Http\Requests\LanguageRequest;
use Survey\Language\Interfaces\LanguageRepositoryInterface;
use Survey\Language\Models\Language;
use Survey\Language\Forms\Language as Form;

/**
 * APIController  class for language.
 */
class LanguageAPIController extends BaseController
{

    /**
     * Initialize language resource controller.
     *
     * @param type LanguageRepositoryInterface $language
     *
     * @return null
     */
    public function __construct(LanguageRepositoryInterface $language)
    {
        parent::__construct();
        $this->repository = $language;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Survey\Language\Repositories\Criteria\LanguageResourceCriteria::class);
    }

    /**
     * Display a list of language.
     *
     * @return Response
     */
    public function index(LanguageRequest $request)
    {
        return $this->repository
            ->setPresenter(\Survey\Language\Repositories\Presenter\LanguagePresenter::class)
            ->paginate();
    }

    /**
     * Display language.
     *
     * @param Request $request
     * @param Model   $language
     *
     * @return Response
     */
    public function show(LanguageRequest $request, Language $language)
    {
        return $language->setPresenter(\Survey\Language\Repositories\Presenter\LanguageListPresenter::class);
        ;
    }

    /**
     * Create new language.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(LanguageRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('language::language.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('language/language/' . $language->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('language/language');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the language.
     *
     * @param Request $request
     * @param Model   $language
     *
     * @return Response
     */
    public function update(LanguageRequest $request, Language $language)
    {
        try {
            $data = $request->all();

            $language->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('language::language.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('language/language/' . $language->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('language/language/' . $language->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the language.
     *
     * @param Model   $language
     *
     * @return Response
     */
    public function destroy(LanguageRequest $request, Language $language)
    {
        try {
            $language->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('language::language.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('language/language/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('language/language/' . $language->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
