<?php

namespace Litecms\Form\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Litecms\Form\Http\Requests\FormRequest;
use Litecms\Form\Interfaces\FormRepositoryInterface;
use Litecms\Form\Models\Form;

/**
 * Resource controller class for form.
 */
class FormResourceController extends BaseController
{

    /**
     * Initialize form resource controller.
     *
     * @param type FormRepositoryInterface $form
     *
     * @return null
     */
    public function __construct(FormRepositoryInterface $form)
    {
        parent::__construct();
        $this->repository = $form;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Litecms\Form\Repositories\Criteria\FormResourceCriteria::class);
    }

    /**
     * Display a list of form.
     *
     * @return Response
     */
    public function index(FormRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Litecms\Form\Repositories\Presenter\FormPresenter::class)
                ->$function();
        }

        $forms = $this->repository->paginate();

        return $this->response->title(trans('form::form.names'))
            ->view('form::form.index', true)
            ->data(compact('forms'))
            ->output();
    }

    /**
     * Display form.
     *
     * @param Request $request
     * @param Model   $form
     *
     * @return Response
     */
    public function show(FormRequest $request, Form $form)
    {

        if ($form->exists) {
            $view = 'form::form.show';
        } else {
            $view = 'form::form.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('form::form.name'))
            ->data(compact('form'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new form.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(FormRequest $request)
    {

        $form = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('form::form.name')) 
            ->view('form::form.create', true) 
            ->data(compact('form'))
            ->output();
    }

    /**
     * Create new form.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(FormRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $form                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('form::form.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('form/form/' . $form->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/form/form'))
                ->redirect();
        }

    }

    /**
     * Show form for editing.
     *
     * @param Request $request
     * @param Model   $form
     *
     * @return Response
     */
    public function edit(FormRequest $request, Form $form)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('form::form.name'))
            ->view('form::form.edit', true)
            ->data(compact('form'))
            ->output();
    }

    /**
     * Update the form.
     *
     * @param Request $request
     * @param Model   $form
     *
     * @return Response
     */
    public function update(FormRequest $request, Form $form)
    {
        try {
            $attributes = $request->all();

            $form->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('form::form.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('form/form/' . $form->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('form/form/' . $form->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the form.
     *
     * @param Model   $form
     *
     * @return Response
     */
    public function destroy(FormRequest $request, Form $form)
    {
        try {

            $form->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('form::form.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('form/form/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('form/form/' . $form->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple form.
     *
     * @param Model   $form
     *
     * @return Response
     */
    public function delete(FormRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('form::form.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('form/form'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/form/form'))
                ->redirect();
        }

    }

    /**
     * Restore deleted forms.
     *
     * @param Model   $form
     *
     * @return Response
     */
    public function restore(FormRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('form::form.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/form/form'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/form/form/'))
                ->redirect();
        }

    }

}
