<?php

namespace Bixo\Demo\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Bixo\Demo\Http\Requests\UserRequest;
use Bixo\Demo\Interfaces\UserRepositoryInterface;
use Bixo\Demo\Models\User;

/**
 * Resource controller class for user.
 */
class UserResourceController extends BaseController
{

    /**
     * Initialize user resource controller.
     *
     * @param type UserRepositoryInterface $user
     *
     * @return null
     */
    public function __construct(UserRepositoryInterface $user)
    {
        parent::__construct();
        $this->repository = $user;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Demo\Repositories\Criteria\UserResourceCriteria::class);
    }

    /**
     * Display a list of user.
     *
     * @return Response
     */
    public function index(UserRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Bixo\Demo\Repositories\Presenter\UserPresenter::class)
                ->$function();
        }

        $users = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('demo::user.names'))
            ->view('demo::user.index', true)
            ->data(compact('users', 'view'))
            ->output();
    }

    /**
     * Display user.
     *
     * @param Request $request
     * @param Model   $user
     *
     * @return Response
     */
    public function show(UserRequest $request, User $user)
    {

        if ($user->exists) {
            $view = 'demo::user.show';
        } else {
            $view = 'demo::user.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('demo::user.name'))
            ->data(compact('user'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new user.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(UserRequest $request)
    {

        $user = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('demo::user.name')) 
            ->view('demo::user.create', true) 
            ->data(compact('user'))
            ->output();
    }

    /**
     * Create new user.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(UserRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $user                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('demo::user.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('demo/user/' . $user->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/demo/user'))
                ->redirect();
        }

    }

    /**
     * Show user for editing.
     *
     * @param Request $request
     * @param Model   $user
     *
     * @return Response
     */
    public function edit(UserRequest $request, User $user)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('demo::user.name'))
            ->view('demo::user.edit', true)
            ->data(compact('user'))
            ->output();
    }

    /**
     * Update the user.
     *
     * @param Request $request
     * @param Model   $user
     *
     * @return Response
     */
    public function update(UserRequest $request, User $user)
    {
        try {
            $attributes = $request->all();

            $user->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('demo::user.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('demo/user/' . $user->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('demo/user/' . $user->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the user.
     *
     * @param Model   $user
     *
     * @return Response
     */
    public function destroy(UserRequest $request, User $user)
    {
        try {

            $user->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('demo::user.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('demo/user/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('demo/user/' . $user->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple user.
     *
     * @param Model   $user
     *
     * @return Response
     */
    public function delete(UserRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('demo::user.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('demo/user'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/demo/user'))
                ->redirect();
        }

    }

    /**
     * Restore deleted users.
     *
     * @param Model   $user
     *
     * @return Response
     */
    public function restore(UserRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('demo::user.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/demo/user'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/demo/user/'))
                ->redirect();
        }

    }

}
