<?php

namespace Bixo\Demo\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Bixo\Demo\Http\Requests\UserRequest;
use Bixo\Demo\Interfaces\UserRepositoryInterface;
use Bixo\Demo\Models\User;
use Bixo\Demo\Forms\User as Form;

/**
 * APIController  class for user.
 */
class UserAPIController extends BaseController
{

    /**
     * Initialize user resource controller.
     *
     * @param type UserRepositoryInterface $user
     *
     * @return null
     */
    public function __construct(UserRepositoryInterface $user)
    {
        parent::__construct();
        $this->repository = $user;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Demo\Repositories\Criteria\UserResourceCriteria::class);
    }

    /**
     * Display a list of user.
     *
     * @return Response
     */
    public function index(UserRequest $request)
    {
        return $this->repository
            ->setPresenter(\Bixo\Demo\Repositories\Presenter\UserPresenter::class)
            ->paginate();
    }

    /**
     * Display user.
     *
     * @param Request $request
     * @param Model   $user
     *
     * @return Response
     */
    public function show(UserRequest $request, User $user)
    {
        return $user->setPresenter(\Bixo\Demo\Repositories\Presenter\UserListPresenter::class);
        ;
    }

    /**
     * Create new user.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(UserRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('demo::user.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('demo/user/' . $user->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('demo/user');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the user.
     *
     * @param Request $request
     * @param Model   $user
     *
     * @return Response
     */
    public function update(UserRequest $request, User $user)
    {
        try {
            $data = $request->all();

            $user->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('demo::user.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('demo/user/' . $user->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('demo/user/' . $user->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the user.
     *
     * @param Model   $user
     *
     * @return Response
     */
    public function destroy(UserRequest $request, User $user)
    {
        try {
            $user->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('demo::user.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('demo/user/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('demo/user/' . $user->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
