<?php
namespace Litecms\Blog\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\View;
use Litecms\Blog\Events\BlogWorkflow as BlogWorkflowEvent;
use Illuminate\Support\Facades\URL;

class BlogWorkflow extends Notification implements ShouldQueue
{
    use Queueable;

    protected $event;
    protected $blog;
    protected $transition;
    protected $request;
    protected $transitions;

    /**
     * Create a new notification instance.
     *
     * @param BlogWorkflowEvent $event
     */
    public function __construct(BlogWorkflowEvent $event)
    {
        $this->transition = $event->transition;
        $this->blog = $event->blog;
        $this->request = $event->request;
        $this->event = $event;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param   mixed  $notifiable
     * @return MailMessage
     */
    public function toMail($notifiable)
    {
        // Set the mail message subject and greeting
        return (new MailMessage)
            ->subject($this->toArray($notifiable)['subject'])
            ->greeting($this->toArray($notifiable)['message'])
            ->action('View Details', $this->url($notifiable));
    }

    /**
     * Get the array representation of the notification.
     *
     * @param   mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        // Set the notification message subject, message, and URL
        return [
            'subject' => trans('blog::blog.notification.' . $this->transition . '.subject') . ' - ' . config('app.name'),
            'message' => trans('blog::blog.notification.' . $this->transition . '.message'),
            'url' => $this->url($notifiable),
        ];
    }

    /**
     * Get the database representation of the notification.
     *
     * @param   mixed  $notifiable
     * @return DatabaseMessage
     */
    public function toDatabase($notifiable)
    {
        // Set the database message subject, message, and transitions
        return new DatabaseMessage([
            'subject' => $this->toArray($notifiable)['subject'],
            'message' => $this->toArray($notifiable)['message'],
            'url' => $this->url($notifiable),
        ]);
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        // Set the notification delivery channels
        return ['mail', 'database'];
    }

    /**
     * Get the temporary signed URL for the notification.
     *
     * @param   mixed  $notifiable
     * @return string
     */
    private function url($notifiable)
    {
        $url = URL::temporarySignedRoute(
            'litecms.blog.workflow', now()->addHours(6), [
                'guard' => 'admin',
                'blog' => $this->blog->getSignedId(now()->addHours(6)),
                'user' => $notifiable->getSignedId(now()->addHours(6)),
                'trans' => 'en',
            ]
        );
        return $url;
    }

    /**
     * Get the view for the notification.
     *
     * @return string
     */
    private function view()
    {
        if (View::exists('blog::blog.notification.' . $this->transition)) {
            return 'blog::blog.notification.' . $this->transition;
        }
        return 'blog::blog.notification.workflow';
    }

}
