<?php

namespace Litecms\Blog\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Litecms\Blog\Events\TagWorkflow as TagWorkflowEvent;
use Litecms\Blog\Notifications\TagWorkflow as TagWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class TagWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(TagWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $tag is submitted.
     *
     * @param TagWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(TagWorkflowEvent $event)
    {
        $client = $event->tag->client;
        Notification::send($client, new TagWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $tag is published.
     *
     * @param TagWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(TagWorkflowEvent $event)
    {
        $client = $event->tag->client;
        Notification::send($client, new TagWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $tag is approved.
     *
     * @param TagWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(TagWorkflowEvent $event)
    {
        $client = $event->tag->client;
        Notification::send($client, new TagWorkflowNotification($event));
    }

    /**
     * Handles the $tag workflow event as a listener.
     *
     * @param TagWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(TagWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
