<?php

namespace Litecms\Blog\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Litecms\Blog\Events\BlogWorkflow as BlogWorkflowEvent;
use Litecms\Blog\Notifications\BlogWorkflow as BlogWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class BlogWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(BlogWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $blog is submitted.
     *
     * @param BlogWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(BlogWorkflowEvent $event)
    {
        $client = $event->blog->client;
        Notification::send($client, new BlogWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $blog is published.
     *
     * @param BlogWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(BlogWorkflowEvent $event)
    {
        $client = $event->blog->client;
        Notification::send($client, new BlogWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $blog is approved.
     *
     * @param BlogWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(BlogWorkflowEvent $event)
    {
        $client = $event->blog->client;
        Notification::send($client, new BlogWorkflowNotification($event));
    }

    /**
     * Handles the $blog workflow event as a listener.
     *
     * @param BlogWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(BlogWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
