<?php

namespace Litecms\Blog\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litecms\Blog\Forms\Tag as TagForm;
use Litecms\Blog\Http\Requests\TagResourceRequest;
use Litecms\Blog\Http\Resources\TagResource;
use Litecms\Blog\Http\Resources\TagsCollection;
use Litecms\Blog\Models\Tag;
use Litecms\Blog\Actions\TagAction;
use Litecms\Blog\Actions\TagActions;


/**
 * Resource controller class for tag.
 */
class TagResourceController extends BaseController
{

    /**
     * Initialize tag resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = TagForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('litecms.blog.modules'), 'blog', guard_url('blog'));
            return $next($request);
        });
    }

    /**
     * Display a list of tag.
     *
     * @return Response
     */
    public function index(TagResourceRequest $request)
    {
        $request = $request->all();
        $page = TagActions::run('paginate', $request);

        $data = new TagsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('blog::tag.names'))
            ->view('blog::tag.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display tag.
     *
     * @param Request $request
     * @param Model   $tag
     *
     * @return Response
     */
    public function show(TagResourceRequest $request, Tag $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new TagResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('blog::tag.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('blog::tag.show')
            ->output();
    }

    /**
     * Show the form for creating a new tag.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(TagResourceRequest $request, Tag $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new TagResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('blog::tag.name'))
            ->view('blog::tag.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new tag.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(TagResourceRequest $request, Tag $model)
    {
        try {
            $request = $request->all();
            $model = TagAction::run('store', $model, $request);
            $data = new TagResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('blog::tag.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('blog/tag/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/blog/tag'))
                ->redirect();
        }

    }

    /**
     * Show tag for editing.
     *
     * @param Request $request
     * @param Model   $tag
     *
     * @return Response
     */
    public function edit(TagResourceRequest $request, Tag $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new TagResource($model);
        // return view('blog::tag.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('blog::tag.name'))
            ->view('blog::tag.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the tag.
     *
     * @param Request $request
     * @param Model   $tag
     *
     * @return Response
     */
    public function update(TagResourceRequest $request, Tag $model)
    {
        try {
            $request = $request->all();
            $model = TagAction::run('update', $model, $request);
            $data = new TagResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('blog::tag.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('blog/tag/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('blog/tag/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the tag.
     *
     * @param Model   $tag
     *
     * @return Response
     */
    public function destroy(TagResourceRequest $request, Tag $model)
    {
        try {

            $request = $request->all();
            $model = TagAction::run('destroy', $model, $request);
            $data = new TagResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('blog::tag.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('blog/tag/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('blog/tag/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
