<?php

namespace Litecms\Blog\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litecms\Blog\Forms\Category as CategoryForm;
use Litecms\Blog\Http\Requests\CategoryResourceRequest;
use Litecms\Blog\Http\Resources\CategoryResource;
use Litecms\Blog\Http\Resources\CategoriesCollection;
use Litecms\Blog\Models\Category;
use Litecms\Blog\Actions\CategoryAction;
use Litecms\Blog\Actions\CategoryActions;


/**
 * Resource controller class for category.
 */
class CategoryResourceController extends BaseController
{

    /**
     * Initialize category resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = CategoryForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('litecms.blog.modules'), 'blog', guard_url('blog'));
            return $next($request);
        });
    }

    /**
     * Display a list of category.
     *
     * @return Response
     */
    public function index(CategoryResourceRequest $request)
    {
        $request = $request->all();
        $page = CategoryActions::run('paginate', $request);

        $data = new CategoriesCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('blog::category.names'))
            ->view('blog::category.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display category.
     *
     * @param Request $request
     * @param Model   $category
     *
     * @return Response
     */
    public function show(CategoryResourceRequest $request, Category $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new CategoryResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('blog::category.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('blog::category.show')
            ->output();
    }

    /**
     * Show the form for creating a new category.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(CategoryResourceRequest $request, Category $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new CategoryResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('blog::category.name'))
            ->view('blog::category.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new category.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(CategoryResourceRequest $request, Category $model)
    {
        try {
            $request = $request->all();
            $model = CategoryAction::run('store', $model, $request);
            $data = new CategoryResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('blog::category.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('blog/category/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/blog/category'))
                ->redirect();
        }

    }

    /**
     * Show category for editing.
     *
     * @param Request $request
     * @param Model   $category
     *
     * @return Response
     */
    public function edit(CategoryResourceRequest $request, Category $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new CategoryResource($model);
        // return view('blog::category.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('blog::category.name'))
            ->view('blog::category.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the category.
     *
     * @param Request $request
     * @param Model   $category
     *
     * @return Response
     */
    public function update(CategoryResourceRequest $request, Category $model)
    {
        try {
            $request = $request->all();
            $model = CategoryAction::run('update', $model, $request);
            $data = new CategoryResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('blog::category.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('blog/category/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('blog/category/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the category.
     *
     * @param Model   $category
     *
     * @return Response
     */
    public function destroy(CategoryResourceRequest $request, Category $model)
    {
        try {

            $request = $request->all();
            $model = CategoryAction::run('destroy', $model, $request);
            $data = new CategoryResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('blog::category.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('blog/category/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('blog/category/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
