<?php

namespace Litecms\Blog\Http\Controllers;

use Exception;
use Litecms\Blog\Actions\BlogWorkflow;
use Litecms\Blog\Http\Requests\BlogWorkflowRequest;
use Litecms\Blog\Http\Resources\BlogResource;
use Litecms\Blog\Models\Blog;
use Litepie\Http\Controllers\ActionController as BaseController;

/**
 * Workflow controller class.
 *
 */
class BlogWorkflowController extends BaseController
{
    /**
     * Action controller function for blog.
     *
     * @param Model $blog
     * @param action next action for the blog.
     *
     * @return Response
     */
    public function __invoke(BlogWorkflowRequest $request, Blog $blog, $trasition)
    {
        try {
            $request = $request->all();
            $blog = BlogWorkflow::run($trasition, $blog, $request);
            $data = new BlogResource($blog);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('blog::blog.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('blog/blog/' . $blog->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('blog/blog/' . $blog->getRouteKey()))
                ->redirect();
        }
    }


    /**
     * Get the blog and user for the workflow.
     *
     * @param PublicRequest $request
     * @param string $blog
     * @param string $user
     *
     * @return Response
     */
    public function get(PublicRequest $request, $blog, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }

        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $blog = Blog::findBySignedId($blog);
        $transitions = $this->transitions($blog, $user);
        $user = user();
        Auth::logout();
        return $this->response->setMetaTitle(trans('blog::blog.name') . ' ' . trans('Approvals'))
            ->view('blog::public.blog.workflow')
            ->layout('mobile')
            ->data(compact('blog', 'user', 'transitions'))
            ->output();
    }

    /**
     * Display the workflow form for the blog.
     *
     * @param PublicRequest $request
     * @param string $blog
     * @param string $user
     *
     * @return Response
     */
    public function post(PublicRequest $request, $blog, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }
        $transition = $request->transition;
        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $blog = Blog::findBySignedId($blog);
        $user = user();
        $blog = BlogWorkflow::run($transition, $blog, $request->all());
        Auth::logout();

        return response()->json(
            [
                'status' => 'success',
                'url' => $request->url(),
            ]
        );
    }

    private function transitions($blog, $user)
    {
        $transitions = [];
        foreach ($blog->workflow()->transitions($blog) as $key => $value) {
            $name = $value->getName();
            $array['url'] = URL::temporarySignedRoute('litecms.blog.workflow', now()->addMinutes(3), [
                'guard' => 'admin',
                'transition' => $name,
                'blog' => $blog->getSignedId(now()->addMinutes(3)),
                'user' => $user->getSignedId(now()->addMinutes(3)),
                'trans' => 'en',
            ]
            );
            $array['name'] = $name;
            $array['key'] = $name;
            $array['form'] = $blog->workflow()->form($value);
            $array['label'] = trans('blog::blog.workflow.' . $name);
            $transitions[] = $array;
        }
        return $transitions;
    }
}
