<?php

namespace Litecms\Blog\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litecms\Blog\Forms\Blog as BlogForm;
use Litecms\Blog\Http\Requests\BlogResourceRequest;
use Litecms\Blog\Http\Resources\BlogResource;
use Litecms\Blog\Http\Resources\BlogsCollection;
use Litecms\Blog\Models\Blog;
use Litecms\Blog\Actions\BlogAction;
use Litecms\Blog\Actions\BlogActions;


/**
 * Resource controller class for blog.
 */
class BlogResourceController extends BaseController
{

    /**
     * Initialize blog resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = BlogForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('litecms.blog.modules'), 'blog', guard_url('blog'));
            return $next($request);
        });
    }

    /**
     * Display a list of blog.
     *
     * @return Response
     */
    public function index(BlogResourceRequest $request)
    {
        $request = $request->all();
        $page = BlogActions::run('paginate', $request);

        $data = new BlogsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('blog::blog.names'))
            ->view('blog::blog.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display blog.
     *
     * @param Request $request
     * @param Model   $blog
     *
     * @return Response
     */
    public function show(BlogResourceRequest $request, Blog $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new BlogResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('blog::blog.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('blog::blog.show')
            ->output();
    }

    /**
     * Show the form for creating a new blog.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(BlogResourceRequest $request, Blog $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new BlogResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('blog::blog.name'))
            ->view('blog::blog.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new blog.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(BlogResourceRequest $request, Blog $model)
    {
        try {
            $request = $request->all();
            $model = BlogAction::run('store', $model, $request);
            $data = new BlogResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('blog::blog.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('blog/blog/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/blog/blog'))
                ->redirect();
        }

    }

    /**
     * Show blog for editing.
     *
     * @param Request $request
     * @param Model   $blog
     *
     * @return Response
     */
    public function edit(BlogResourceRequest $request, Blog $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new BlogResource($model);
        // return view('blog::blog.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('blog::blog.name'))
            ->view('blog::blog.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the blog.
     *
     * @param Request $request
     * @param Model   $blog
     *
     * @return Response
     */
    public function update(BlogResourceRequest $request, Blog $model)
    {
        try {
            $request = $request->all();
            $model = BlogAction::run('update', $model, $request);
            $data = new BlogResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('blog::blog.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('blog/blog/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('blog/blog/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the blog.
     *
     * @param Model   $blog
     *
     * @return Response
     */
    public function destroy(BlogResourceRequest $request, Blog $model)
    {
        try {

            $request = $request->all();
            $model = BlogAction::run('destroy', $model, $request);
            $data = new BlogResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('blog::blog.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('blog/blog/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('blog/blog/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
