<?php

namespace Litecms\Calllist\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Repository\Filter\RequestFilter;
use Litecms\Calllist\Forms\Calllist as CalllistForm;
use Litecms\Calllist\Http\Requests\CalllistRequest;
use Litecms\Calllist\Interfaces\CalllistRepositoryInterface;
use Litecms\Calllist\Repositories\Eloquent\Filters\CalllistResourceFilter;
use Litecms\Calllist\Repositories\Eloquent\Presenters\CalllistListPresenter;

/**
 * Resource controller class for calllist.
 */
class CalllistResourceController extends BaseController
{

    /**
     * Initialize calllist resource controller.
     *
     *
     * @return null
     */
    public function __construct(CalllistRepositoryInterface $calllist)
    {
        parent::__construct();
        $this->form = CalllistForm::setAttributes()->toArray();
        $this->modules = $this->modules(config('litecms.calllist.modules'), 'calllist', guard_url('calllist'));
        $this->repository = $calllist;
    }

    /**
     * Display a list of calllist.
     *
     * @return Response
     */
    public function index(CalllistRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $data = $this->repository
            ->pushFilter(RequestFilter::class)
            ->pushFilter(CalllistResourceFilter::class)
            ->setPresenter(CalllistListPresenter::class)
            ->simplePaginate($pageLimit)
            // ->withQueryString()
            ->toArray();

        extract($data);
        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('calllist::calllist.names'))
            ->view('calllist::calllist.index')
            ->data(compact('data', 'meta', 'links', 'modules', 'form'))
            ->output();
    }

    /**
     * Display calllist.
     *
     * @param Request $request
     * @param Model   $calllist
     *
     * @return Response
     */
    public function show(CalllistRequest $request, CalllistRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('calllist::calllist.name'))
            ->data(compact('data', 'form', 'modules', 'form'))
            ->view('calllist::calllist.show')
            ->output();
    }

    /**
     * Show the form for creating a new calllist.
     *
     * @param Request $request
     *x
     * @return Response
     */
    public function create(CalllistRequest $request, CalllistRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('calllist::calllist.name'))
            ->view('calllist::calllist.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Create new calllist.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(CalllistRequest $request, CalllistRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $repository->create($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.created', ['Module' => trans('calllist::calllist.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('calllist/calllist/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/calllist/calllist'))
                ->redirect();
        }

    }

    /**
     * Show calllist for editing.
     *
     * @param Request $request
     * @param Model   $calllist
     *
     * @return Response
     */
    public function edit(CalllistRequest $request, CalllistRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('calllist::calllist.name'))
            ->view('calllist::calllist.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Update the calllist.
     *
     * @param Request $request
     * @param Model   $calllist
     *
     * @return Response
     */
    public function update(CalllistRequest $request, CalllistRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $repository->update($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('calllist::calllist.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('calllist/calllist/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('calllist/calllist/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the calllist.
     *
     * @param Model   $calllist
     *
     * @return Response
     */
    public function destroy(CalllistRequest $request, CalllistRepositoryInterface $repository)
    {
        try {
            $repository->delete();
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('calllist::calllist.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('calllist/calllist/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('calllist/calllist/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }
}
